<?php
/**
 * Matomo - free/libre analytics platform
 *
 * @link https://matomo.org
 * @license http://www.gnu.org/licenses/gpl-3.0.html GPL v3 or later
 */

namespace Piwik\Plugins\WeatherWidget\Widgets;

use Piwik\Http;
use Piwik\Option;
use Piwik\Piwik;
use Piwik\Plugins\WeatherWidget\SystemSettings;
use Piwik\Widget\Widget;
use Piwik\Widget\WidgetConfig;

class Weather extends Widget
{
    private const CACHE_KEY = 'WeatherWidget_cachedData';
    private const CACHE_TTL = 1800; // 30 minutes

    // WMO weather code => [translation key, icon]
    private const WEATHER_CODES = [
        0 => ['key' => 'WeatherWidget_WeatherClear', 'icon' => "\u{2600}\u{FE0F}"],
        1 => ['key' => 'WeatherWidget_WeatherMainlyClear', 'icon' => "\u{1F324}\u{FE0F}"],
        2 => ['key' => 'WeatherWidget_WeatherPartlyCloudy', 'icon' => "\u{26C5}"],
        3 => ['key' => 'WeatherWidget_WeatherOvercast', 'icon' => "\u{2601}\u{FE0F}"],
        45 => ['key' => 'WeatherWidget_WeatherFog', 'icon' => "\u{1F32B}\u{FE0F}"],
        48 => ['key' => 'WeatherWidget_WeatherRimeFog', 'icon' => "\u{1F32B}\u{FE0F}"],
        51 => ['key' => 'WeatherWidget_WeatherLightDrizzle', 'icon' => "\u{1F327}\u{FE0F}"],
        53 => ['key' => 'WeatherWidget_WeatherModerateDrizzle', 'icon' => "\u{1F327}\u{FE0F}"],
        55 => ['key' => 'WeatherWidget_WeatherDenseDrizzle', 'icon' => "\u{1F327}\u{FE0F}"],
        56 => ['key' => 'WeatherWidget_WeatherLightFreezingDrizzle', 'icon' => "\u{1F327}\u{FE0F}"],
        57 => ['key' => 'WeatherWidget_WeatherDenseFreezingDrizzle', 'icon' => "\u{1F327}\u{FE0F}"],
        61 => ['key' => 'WeatherWidget_WeatherLightRain', 'icon' => "\u{1F327}\u{FE0F}"],
        63 => ['key' => 'WeatherWidget_WeatherModerateRain', 'icon' => "\u{1F327}\u{FE0F}"],
        65 => ['key' => 'WeatherWidget_WeatherHeavyRain', 'icon' => "\u{1F327}\u{FE0F}"],
        66 => ['key' => 'WeatherWidget_WeatherLightFreezingRain', 'icon' => "\u{1F327}\u{FE0F}"],
        67 => ['key' => 'WeatherWidget_WeatherHeavyFreezingRain', 'icon' => "\u{1F327}\u{FE0F}"],
        71 => ['key' => 'WeatherWidget_WeatherLightSnow', 'icon' => "\u{1F328}\u{FE0F}"],
        73 => ['key' => 'WeatherWidget_WeatherModerateSnow', 'icon' => "\u{1F328}\u{FE0F}"],
        75 => ['key' => 'WeatherWidget_WeatherHeavySnow', 'icon' => "\u{2744}\u{FE0F}"],
        77 => ['key' => 'WeatherWidget_WeatherSnowGrains', 'icon' => "\u{1F328}\u{FE0F}"],
        80 => ['key' => 'WeatherWidget_WeatherLightShowers', 'icon' => "\u{1F326}\u{FE0F}"],
        81 => ['key' => 'WeatherWidget_WeatherModerateShowers', 'icon' => "\u{1F326}\u{FE0F}"],
        82 => ['key' => 'WeatherWidget_WeatherViolentShowers', 'icon' => "\u{26C8}\u{FE0F}"],
        85 => ['key' => 'WeatherWidget_WeatherLightSnowShowers', 'icon' => "\u{1F328}\u{FE0F}"],
        86 => ['key' => 'WeatherWidget_WeatherHeavySnowShowers', 'icon' => "\u{1F328}\u{FE0F}"],
        95 => ['key' => 'WeatherWidget_WeatherThunderstorm', 'icon' => "\u{26C8}\u{FE0F}"],
        96 => ['key' => 'WeatherWidget_WeatherThunderstormLightHail', 'icon' => "\u{26C8}\u{FE0F}"],
        99 => ['key' => 'WeatherWidget_WeatherThunderstormHeavyHail', 'icon' => "\u{26C8}\u{FE0F}"],
    ];

    /** @var SystemSettings */
    private $settings;

    public function __construct(SystemSettings $settings)
    {
        $this->settings = $settings;
    }

    public static function configure(WidgetConfig $config)
    {
        $config->setCategoryId('WeatherWidget_Weather');
        $config->setName('WeatherWidget_TodayWeather');
        $config->setOrder(1);
    }

    public function render()
    {
        try {
            $weatherData = $this->getWeatherData();
            return $this->renderWeatherWidget($weatherData);
        } catch (\Exception $e) {
            return $this->renderError($e->getMessage());
        }
    }

    private function getWeatherData(): array
    {
        $cached = $this->getCachedData();
        if ($cached !== null) {
            return $cached;
        }

        try {
            $data = $this->fetchWeatherData();
            $this->setCachedData($data);
            return $data;
        } catch (\Exception $e) {
            // Serve stale cache on API failure
            $stale = $this->getCachedData(true);
            if ($stale !== null) {
                return $stale;
            }
            throw $e;
        }
    }

    private function getCachedData(bool $ignoreExpiry = false): ?array
    {
        $raw = Option::get(self::CACHE_KEY);
        if ($raw === false) {
            return null;
        }

        $cached = json_decode($raw, true);
        if (!is_array($cached) || !isset($cached['data'], $cached['timestamp'])) {
            return null;
        }

        if (!$ignoreExpiry && (time() - $cached['timestamp']) > self::CACHE_TTL) {
            return null;
        }

        return $cached['data'];
    }

    private function setCachedData(array $data): void
    {
        $payload = json_encode([
            'data' => $data,
            'timestamp' => time(),
        ]);
        Option::set(self::CACHE_KEY, $payload);
    }

    private function fetchWeatherData(): array
    {
        $latitude = $this->settings->latitude->getValue();
        $longitude = $this->settings->longitude->getValue();
        $timezone = $this->settings->timezone->getValue();
        $units = $this->settings->units->getValue();

        $tempUnit = ($units === 'imperial') ? 'fahrenheit' : 'celsius';
        $windUnit = ($units === 'imperial') ? 'mph' : 'kmh';

        $url = sprintf(
            'https://api.open-meteo.com/v1/forecast?latitude=%s&longitude=%s&current=temperature_2m,relative_humidity_2m,weather_code,wind_speed_10m&timezone=%s&temperature_unit=%s&wind_speed_unit=%s',
            urlencode($latitude),
            urlencode($longitude),
            urlencode($timezone),
            urlencode($tempUnit),
            urlencode($windUnit)
        );

        $response = Http::sendHttpRequest($url, 5);
        $data = json_decode($response, true);

        if (!$data || !isset($data['current'])) {
            throw new \Exception(Piwik::translate('WeatherWidget_ErrorFetchingData'));
        }

        return $data['current'];
    }

    private function getWeatherInfo(int $code): array
    {
        $entry = self::WEATHER_CODES[$code] ?? null;
        if ($entry) {
            return [
                'description' => Piwik::translate($entry['key']),
                'icon' => $entry['icon'],
            ];
        }

        return [
            'description' => Piwik::translate('WeatherWidget_WeatherUnknown'),
            'icon' => "\u{2753}",
        ];
    }

    private function renderWeatherWidget(array $data): string
    {
        $weatherCode = (int)($data['weather_code'] ?? 0);
        $weatherInfo = $this->getWeatherInfo($weatherCode);
        $temperature = round($data['temperature_2m'] ?? 0);
        $humidity = $data['relative_humidity_2m'] ?? 0;
        $windSpeed = round($data['wind_speed_10m'] ?? 0);

        $units = $this->settings->units->getValue();
        $tempUnit = ($units === 'imperial') ? "\u{00B0}F" : "\u{00B0}C";
        $windUnit = ($units === 'imperial') ? 'mph' : 'km/h';
        $cityName = $this->settings->cityName->getValue();

        $html = <<<'HTML'
<div style="padding: 20px; text-align: center; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;">
    <div style="font-size: 14px; color: var(--text-color-lighter, #666); margin-bottom: 10px;">
        %s %s
    </div>
    <div style="font-size: 64px; margin: 10px 0;">
        %s
    </div>
    <div style="font-size: 36px; font-weight: bold; color: var(--text-color-default, #333); margin: 10px 0;">
        %d%s
    </div>
    <div style="font-size: 16px; color: var(--text-color-light, #555); margin-bottom: 15px;">
        %s
    </div>
    <div style="display: flex; justify-content: center; gap: 30px; font-size: 14px; color: var(--text-color-lighter, #666);">
        <div>
            %s %s: %d%%
        </div>
        <div>
            %s %s: %d %s
        </div>
    </div>
    <div style="margin-top: 15px; font-size: 11px; color: var(--text-color-lightest, #999);">
        %s: Open-Meteo.com
    </div>
</div>
HTML;

        return sprintf(
            $html,
            "\u{1F4CD}",
            htmlspecialchars($cityName),
            $weatherInfo['icon'],
            $temperature,
            $tempUnit,
            htmlspecialchars($weatherInfo['description']),
            "\u{1F4A7}",
            Piwik::translate('WeatherWidget_Humidity'),
            $humidity,
            "\u{1F4A8}",
            Piwik::translate('WeatherWidget_Wind'),
            $windSpeed,
            htmlspecialchars($windUnit),
            Piwik::translate('WeatherWidget_DataSource')
        );
    }

    private function renderError(string $message): string
    {
        return sprintf(
            '<div style="padding: 30px; text-align: center; color: var(--color-red, #c00);">
                <div style="font-size: 48px;">%s</div>
                <div style="margin-top: 10px;">%s</div>
                <div style="margin-top: 5px; font-size: 12px; color: var(--text-color-lighter, #666);">%s</div>
            </div>',
            "\u{26A0}\u{FE0F}",
            Piwik::translate('WeatherWidget_ErrorTitle'),
            htmlspecialchars($message)
        );
    }
}
