<?php
/**
 * Matomo - free/libre analytics platform
 *
 * @link https://matomo.org
 * @license http://www.gnu.org/licenses/gpl-3.0.html GPL v3 or later
 */

namespace Piwik\Plugins\WeatherWidget;

use Piwik\Piwik;
use Piwik\Settings\FieldConfig;
use Piwik\Settings\Plugin\SystemSetting;

class SystemSettings extends \Piwik\Settings\Plugin\SystemSettings
{
    /** @var SystemSetting */
    public $latitude;

    /** @var SystemSetting */
    public $longitude;

    /** @var SystemSetting */
    public $cityName;

    /** @var SystemSetting */
    public $timezone;

    /** @var SystemSetting */
    public $units;

    protected function init()
    {
        $this->latitude = $this->createLatitudeSetting();
        $this->longitude = $this->createLongitudeSetting();
        $this->cityName = $this->createCityNameSetting();
        $this->timezone = $this->createTimezoneSetting();
        $this->units = $this->createUnitsSetting();
    }

    private function createLatitudeSetting(): SystemSetting
    {
        return $this->makeSetting('latitude', '48.8566', FieldConfig::TYPE_STRING, function (FieldConfig $field) {
            $field->title = Piwik::translate('WeatherWidget_SettingLatitude');
            $field->uiControl = FieldConfig::UI_CONTROL_TEXT;
            $field->description = Piwik::translate('WeatherWidget_SettingLatitudeDescription');
            $field->validate = function ($value) {
                if (!is_numeric($value)) {
                    throw new \Exception(Piwik::translate('WeatherWidget_ValidationLatitude'));
                }
                $val = (float) $value;
                if ($val < -90 || $val > 90) {
                    throw new \Exception(Piwik::translate('WeatherWidget_ValidationLatitude'));
                }
            };
        });
    }

    private function createLongitudeSetting(): SystemSetting
    {
        return $this->makeSetting('longitude', '2.3522', FieldConfig::TYPE_STRING, function (FieldConfig $field) {
            $field->title = Piwik::translate('WeatherWidget_SettingLongitude');
            $field->uiControl = FieldConfig::UI_CONTROL_TEXT;
            $field->description = Piwik::translate('WeatherWidget_SettingLongitudeDescription');
            $field->validate = function ($value) {
                if (!is_numeric($value)) {
                    throw new \Exception(Piwik::translate('WeatherWidget_ValidationLongitude'));
                }
                $val = (float) $value;
                if ($val < -180 || $val > 180) {
                    throw new \Exception(Piwik::translate('WeatherWidget_ValidationLongitude'));
                }
            };
        });
    }

    private function createCityNameSetting(): SystemSetting
    {
        return $this->makeSetting('cityName', 'Paris, France', FieldConfig::TYPE_STRING, function (FieldConfig $field) {
            $field->title = Piwik::translate('WeatherWidget_SettingCityName');
            $field->uiControl = FieldConfig::UI_CONTROL_TEXT;
            $field->description = Piwik::translate('WeatherWidget_SettingCityNameDescription');
            $field->validate = function ($value) {
                if (strlen($value) > 100) {
                    throw new \Exception(Piwik::translate('WeatherWidget_ValidationCityName'));
                }
            };
        });
    }

    private function createTimezoneSetting(): SystemSetting
    {
        return $this->makeSetting('timezone', 'Europe/Paris', FieldConfig::TYPE_STRING, function (FieldConfig $field) {
            $field->title = Piwik::translate('WeatherWidget_SettingTimezone');
            $field->uiControl = FieldConfig::UI_CONTROL_TEXT;
            $field->description = Piwik::translate('WeatherWidget_SettingTimezoneDescription');
            $field->validate = function ($value) {
                if (!in_array($value, \DateTimeZone::listIdentifiers(), true)) {
                    throw new \Exception(Piwik::translate('WeatherWidget_ValidationTimezone'));
                }
            };
        });
    }

    private function createUnitsSetting(): SystemSetting
    {
        return $this->makeSetting('units', 'metric', FieldConfig::TYPE_STRING, function (FieldConfig $field) {
            $field->title = Piwik::translate('WeatherWidget_SettingUnits');
            $field->uiControl = FieldConfig::UI_CONTROL_SINGLE_SELECT;
            $field->availableValues = [
                'metric' => Piwik::translate('WeatherWidget_SettingUnitsMetric'),
                'imperial' => Piwik::translate('WeatherWidget_SettingUnitsImperial'),
            ];
        });
    }
}
