<?php 
/**
 * Plugin Name: Default Segment Applied (Matomo Plugin)
 * Plugin URI: http://plugins.matomo.org/DefaultSegmentApplied
 * Description: Allows users to set a default segment that is automatically applied when viewing reports.
 * Author: Chardonneaur
 * Author URI: https://github.com/Chardonneaur/DefaultSegmentApplied
 * Version: 1.1.0
 */
?><?php

/**
 * Matomo - free/libre analytics platform
 *
 * @link    https://matomo.org
 * @license https://www.gnu.org/licenses/gpl-3.0.html GPL v3 or later
 */

namespace Piwik\Plugins\DefaultSegmentApplied;

use Piwik\Container\StaticContainer;
use Piwik\Log\LoggerInterface;
use Piwik\Piwik;
use Piwik\Plugin;

 
if (defined( 'ABSPATH')
&& function_exists('add_action')) {
    $path = '/matomo/app/core/Plugin.php';
    if (defined('WP_PLUGIN_DIR') && WP_PLUGIN_DIR && file_exists(WP_PLUGIN_DIR . $path)) {
        require_once WP_PLUGIN_DIR . $path;
    } elseif (defined('WPMU_PLUGIN_DIR') && WPMU_PLUGIN_DIR && file_exists(WPMU_PLUGIN_DIR . $path)) {
        require_once WPMU_PLUGIN_DIR . $path;
    } else {
        return;
    }
    add_action('plugins_loaded', function () {
        if (function_exists('matomo_add_plugin')) {
            matomo_add_plugin(__DIR__, __FILE__, true);
        }
    });
}

class DefaultSegmentApplied extends Plugin
{
    public function registerEvents()
    {
        return [
            'Template.jsGlobalVariables'             => 'addJsGlobalVariables',
            'AssetManager.getJavaScriptFiles'        => 'getJsFiles',
            'Translate.getClientSideTranslationKeys' => 'getClientSideTranslationKeys',
        ];
    }

    public function addJsGlobalVariables(&$out)
    {
        if (Piwik::isUserIsAnonymous()) {
            return;
        }

        try {
            $userSettings = new UserSettings();
            $defaultSegment = $userSettings->defaultSegment->getValue();

            // Fallback to system-wide admin default if user has no personal segment
            if (empty($defaultSegment)) {
                $systemSettings = new SystemSettings();
                $defaultSegment = $systemSettings->defaultSegmentForAllUsers->getValue();

                if ($defaultSegment && !$systemSettings->segmentExists($defaultSegment)) {
                    $defaultSegment = '';
                }
            } else {
                if (!$userSettings->segmentExists($defaultSegment)) {
                    $defaultSegment = '';
                }
            }

            $encoded = $defaultSegment ? json_encode($defaultSegment, JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT) : false;
            if ($encoded !== false) {
                $out .= "    piwik.defaultSegment = $encoded;\n";
            } else {
                $out .= "    piwik.defaultSegment = '';\n";
            }
        } catch (\Exception $e) {
            StaticContainer::get(LoggerInterface::class)->warning('DefaultSegmentApplied: failed to load default segment: {message}', ['message' => $e->getMessage()]);
            $out .= "    piwik.defaultSegment = '';\n";
        }
    }

    public function getJsFiles(&$jsFiles)
    {
        $jsFiles[] = 'plugins/DefaultSegmentApplied/javascripts/defaultSegmentApplier.js';
    }

    public function getClientSideTranslationKeys(&$translationKeys)
    {
        $translationKeys[] = 'DefaultSegmentApplied_DefaultSegment';
        $translationKeys[] = 'DefaultSegmentApplied_NoDefaultSegment';
    }
}
